/*******************************************************************************
+
+	generate.c
+
+   Copyright (C) 2000
+	Garrick Welsh, gaz@hons.cs.usyd.edu.au.
+	Kevin Pulo, kev@hons.cs.usyd.edu.au.
+
+	This program is free software; you can redistribute it and/or modify
+	it under the terms of the GNU General Public License as published by
+	the Free Software Foundation; either version 2 of the License, or
+	(at your option) any later version.
+
+	This program is distributed in the hope that it will be useful,
+	but WITHOUT ANY WARRANTY; without even the implied warranty of
+	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
+	GNU General Public License for more details.
+
+	You should have received a copy of the GNU General Public License
+	along with this program; if not, write to the Free Software
+	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
+
+	$Id: generate.c,v 1.6 2000/05/28 13:19:47 kev Exp kev $
+
*******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <assert.h>

/* Warning!!! The functions drand48 and srand48 are used
 * in this program. They are not ansi C compatible
 */


/* The programs usage string */
static char *usage = "Usage: generate [options]\n"
"Options:\n"
"    --help, -help, -h  prints this usage message\n"
"    -r <real>          specifies the radius of the clusters (default 0.01)\n"
"    -k <int>           specifies the number of clusters (default 4)\n"
"    -n <int>           specifies the number of clustered points (default 250)\n"
"    -N <int>           specifies the number of noise points (default 50)\n";

/* The number of random numbers that we want */
static int numbers = 50;
/* The number of numbers we want in clusters */
static int cnumbers = 250;
/* The number of clusters that we want */
static int clusters = 4;

/* The default radii of the clusters that we'll be looking
 * making */
static double radii = 0.01;


typedef struct point { double x, y; } point;

typedef struct arguments { double radii; int k; int n; int numRanPoints; } arguments; 


/** Generate a random number between 0 and 1
 * @return Returns a double which is between 0 and 1
 */
static double getRandomDouble(void)
{
	return drand48();
}

/** This will generate random points over [0,1]x[0,1] and prints them
 * to stdout
 * @param n The number of points we wish to generate
 * @return returns 1 on a sucessful run and 0 on an unsuccessful run
 */
static int generateRandomPoints(int n)
{
	if(n < 1) {
		return 0;
	}else {
		int i;
		double d[2];
		
		for(i = 0; i < n; i++) {
			d[0] = getRandomDouble();
			d[1] = getRandomDouble();
			assert(d[0] < 1.0 && d[1] < 1.0 && d[0] > 0 && d[1] > 0);
			printf("(%10g, %10g)  -1 \n", d[0], d[1]);
		}
	}
	return 1;
}

/** This will generate cluster points over [0,1]x[0,1] and prints them
 * to stdout. It does this by having k circles, we then choose to place
 * each point in a randomly choosen circle.
 * @param n The number of points we wish to generate
 * @param k The number of circles, ie possible clusters
 * @param r the radii of the circles. r must be < 1 and > 0
 * @return returns 1 on a sucessful run and 0 on an unsuccessful run
 */
static int generateClusterPoints(int k, int n, double r)
{
	int i;
	double t;
	int j;
	point *p;

	double gamma, theta, x, y;
	
	assert(r > 0.0 && r < 0.5);

	p = malloc(sizeof(point) * k);
	/* Generate the k possible circles */
	for(i = 0; i < k; i++) {
		p[i].x = getRandomDouble();
		p[i].y = getRandomDouble();
		if((t = p[i].x + r) > 1.0) {
			t -= 1.0;
			p[i].x -= t;
		}
		if((t = p[i].y + r) > 1.0) {
			t -= 1.0;
			p[i].y -= t;
		}
		if((t = p[i].x - r) < 0.0) {
			t = -t;
			p[i].x += t;
		}
		if((t = p[i].y - r) < 0.0) {
			t = -t;
			p[i].y += t;
		}
	}
	
	for(i = 0; i < n; i++) {
		j = rand() % k;
		gamma = getRandomDouble();
		theta = getRandomDouble() * M_PI * 2;
		x = p[j].x + r * gamma * sin(theta);
		y = p[j].y + r * gamma * cos(theta);
    assert(x > 0.0 && x < 1.0 && y > 0.0 && y < 1.0);
		printf("(%10g, %10g) %d\n", x, y, j);
	}
	free(p);
	return 1;
}

void badCommand(void)
{
	fprintf(stderr, "Illegal argument format\n\n%s\n", usage);
	exit(1);
}

void parseCommandLine(arguments *a, int argc, char *argv[])
{
	int i;
	
	a->radii = radii;
	a->k = clusters;
	a->n = cnumbers;
	a->numRanPoints = numbers;
	
	for(i = 1; i < argc; i++) {
		if(!strcmp(argv[i], "--help") || !strcmp(argv[i], "-help") || !strcmp(argv[i], "-h")) {
			printf("%s\n", usage);
			exit(0);
		}

		if(!strcmp(argv[i], "-r")) {
			i++;
			if(i >= argc)
				badCommand();
			if(sscanf(argv[i], "%lg", &(a->radii)) != 1)
				badCommand();

		} else if(!strcmp(argv[i], "-k")) {
			i++;
			if(i >= argc)
				badCommand();
			if(sscanf(argv[i], "%d", &(a->k)) != 1)
				badCommand();

		} else if(!strcmp(argv[i], "-n")) {
			i++;
			if(i >= argc)
				badCommand();
			if(sscanf(argv[i], "%d", &(a->n)) != 1)
				badCommand();

		} else if(!strcmp(argv[i], "-N")) {
			i++;
			if(i >= argc)
				badCommand();
			if(sscanf(argv[i], "%d", &(a->numRanPoints)) != 1)
				badCommand();
		} else {
			badCommand();
		}
	}
}

	
int main(int argc, char *argv[])
{
	int status;
	arguments a;
	status = 0;
	srand(time(NULL));
	srand48(time(NULL));

	parseCommandLine(&a, argc, argv);
	generateRandomPoints(a.numRanPoints);
	generateClusterPoints(a.k, a.n, a.radii);

	return status;
}

